# Boundary Visual Effects - Render layers to Folders
# Version 1.0
#
# For greeting, bugs, and requests email me at mborgo[at]boundaryvfx.com
# Compatibility: Silhouette v4.1 and up, not tested in previous versions
#
# If you like it and use it frequently, please consider a small donation to the author,
# via Paypal on the email mborgo[at]boundaryvfx.com

#===============================================================================
# This action will render all VISIBLE, ROOT layers in the Roto Node, each to its own directory.
##===============================================================================

#===============================================================================
# Version Log
# v1 (2012/11/01)
#===============================================================================

# Copyright (c) 2012, Magno Borgo
# All rights reserved.
#
# BSD-style license:
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#     * Neither the name of Magno Borgo or its contributors may be used to
#       endorse or promote products derived from this software without
#       specific prior written permission.
#
#THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
#AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
#IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
#PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
#BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
#OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
#OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
#OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
#WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
#ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
#EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#===============================================================================
# Instructions to use
#===============================================================================
# Run it from the Actions menu
#===============================================================================


from fx import *
from tools.objectIterator import getObjects
from tools.renderHandler import RenderHandler
from tools.solo import SoloState
import os

class RenderLayersToSeparateFolders(Action):
	"""Renders all visible, root layers in the Roto Node, each to its own directory.
Uses the current session rendering options for output control.
"""

	def __init__(self):
		Action.__init__(self, "BoundaryVFX|Render Layers to Directories")

	def available(self):
		session = activeSession()
		assert session, "Select a Session"
		rotoNode = session.node(type="RotoNode")
		assert rotoNode, "The session does not contain a Roto Node"
		children = rotoNode.children
		objects = getObjects(children, types=[Layer], hidden=False)
		assert len(objects) > 0, "There must be one or more visible layers"

	def execute(self):
		session = activeSession()
		preview = False
		# when running in the gui, clone the session so we can make
		# changes to the object model without worrying about undo
		if gui:
			dialog = RenderDialog(session)
			if not dialog.execute():
				return
			preview = dialog.preview
			session = session.clone()

		session = session.clone()

		rotoNode = session.node(type="RotoNode")
		children = rotoNode.children

		# output only alpha
		session.renderInfo.type = RenderInfo.Output_Alpha

		# find all layers
		objects = getObjects(children, types=[Layer], hidden=False)

		progress = PreviewProgressHandler()
		progress.begin(self.label)
		progress.preview = preview

		baseName = session.renderInfo.file
		name = session.renderInfo.file
		version = ""
		sep = '_v'
		num_pos = name.rfind(sep)
		
		if num_pos >= 0:			
			width = len(name)
			#print width
			name = name[0:num_pos]
			#print "name:",name
			version = baseName[num_pos:width]
			#print "version:", version
			
		separator = ""
		if len(name) > 0:
			separator = "_"
		
		
		


		# solo each layer and render each in its own pass
		sessiondir = session.renderInfo.directory
		
		for object in objects:
			if object.parent.type == "RotoNode":
				state = SoloState(object)
				state.solo()
				## checks for version and place at the end
				session.renderInfo.file = name + separator + object.label.replace(' ', '_') + version
				fullPath = os.path.join(sessiondir, object.label.replace(' ', '_'))
				if not os.path.exists(fullPath):
					print "Creating directory:", fullPath
					os.makedirs(fullPath)
				
				session.renderInfo.directory = fullPath
				# set progress window title to object name
				progress.title = "Rendering %s..." % (object.label)
	
				# create a renderer and target the roto node
				r = RenderHandler(session, progress)
				r.outputNode = rotoNode
				r.type = RenderInfo.Output_Alpha
	
				# render the current project settings
				r.render()
	
				# restore visibility state for the next pass
				state.restore()

		progress.end()

addAction(RenderLayersToSeparateFolders())


